<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Client;
use App\Models\Product;
use App\Models\SaleDetail;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ClientAnalytics extends Component
{
    public $selectedClientId = null;
    public $stats = [];

    // Escuchar evento desde otros componentes para seleccionar cliente
    protected $listeners = ['analyzeClient' => 'loadClientStats'];

    public function render()
    {
        return view('livewire.client-analytics', [
            'clients' => Client::orderBy('name')->get()
        ]);
    }

    public function loadClientStats($clientId)
    {
        $this->selectedClientId = $clientId;
        $this->calculateStats();
    }

    // Si cambian el select manualmente
    public function updatedSelectedClientId() 
    {
        $this->calculateStats();
    }

 public function calculateStats()
{
    if (!$this->selectedClientId) {
        $this->stats = [];
        return;
    }

    // Traemos TODOS los productos activos, no solo los que compró
    $products = Product::where('is_active', true)->get();
    $results = [];

    foreach ($products as $product) {
        // Buscamos historial de este producto para este cliente
        $details = SaleDetail::where('product_id', $product->id)
            ->whereHas('sale', function($q) {
                $q->where('client_id', $this->selectedClientId);
            })
            ->with('sale')
            ->get()
            ->sortBy('sale.sale_date'); // Ordenar del más antiguo al más nuevo para calcular intervalos

        // CASO A: NUNCA HA COMPRADO EL PRODUCTO
        if ($details->isEmpty()) {
            $results[] = [
                'product_name' => $product->name,
                'status' => 'never', // Estado especial
                'last_date' => '-',
                'days_since' => '-',
                'frequency' => '-',
                'alert_message' => 'Oportunidad de Venta'
            ];
            continue; // Pasamos al siguiente producto
        }

        // CASO B: YA LO HA COMPRADO (ANÁLISIS DE FRECUENCIA)
        $lastPurchaseDate = $details->last()->sale->sale_date;
        $firstPurchaseDate = $details->first()->sale->sale_date;
        $totalPurchases = $details->count();
        
        // Días que han pasado desde la última vez que vino
        $daysSinceLast = Carbon::now()->diffInDays($lastPurchaseDate);

        // Calcular Frecuencia Habitual (Gap Promedio)
        // Si solo compró 1 vez, no podemos calcular frecuencia, asumimos 30 días por defecto
        if ($totalPurchases > 1) {
            $totalDaysRange = $firstPurchaseDate->diffInDays($lastPurchaseDate);
            // Promedio de días entre compras (ej: compra cada 7 días)
            $averageGap = $totalDaysRange / ($totalPurchases - 1); 
        } else {
            $averageGap = 30; // Valor por defecto para clientes nuevos con 1 sola compra
        }

        // --- LÓGICA DE ALERTA DINÁMICA ---
        $status = 'active';
        $alertMessage = 'Cliente Activo';

        // Factor de Tolerancia: 3 veces su ciclo habitual
        // Ej: Si compra cada 7 días, nos preocupamos a los 21 días.
        $threshold = $averageGap * 3; 

        if ($daysSinceLast > $threshold) {
            $status = 'danger'; // ALERTA ROJA
            $weeksMissed = round($daysSinceLast / 7);
            $alertMessage = "Hace $weeksMissed semanas no compra (Inusual)";
        } elseif ($daysSinceLast > ($averageGap * 1.5)) {
            $status = 'warning';
            $alertMessage = "Se está retrasando";
        }

        $results[] = [
            'product_name' => $product->name,
            'status' => $status,
            'last_date' => $lastPurchaseDate->format('d/m/Y'),
            'days_since' => $daysSinceLast,
            'frequency' => round($averageGap) . " días", // "Cada 7 días"
            'alert_message' => $alertMessage
        ];
    }

    // Ordenamos: Primero las Alertas Rojas, luego Nunca Comprados, luego los normales
    $this->stats = collect($results)->sortBy(function($row) {
        return match($row['status']) {
            'danger' => 1,
            'never' => 2,
            'warning' => 3,
            default => 4,
        };
    })->values()->all();
}
}